package egovframework.rte.fdl.cryptography.impl;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.math.BigDecimal;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.EncoderException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.util.ReflectionUtils;

import egovframework.rte.fdl.cryptography.EgovARIACryptoService;
import egovframework.rte.fdl.cryptography.EgovPasswordEncoder;

public class EgovARIACryptoServiceImpl implements EgovARIACryptoService {
    private final Base64 base64 = new Base64();
    
    private Logger logger = Logger.getLogger(this.getClass());	// Logger 처리
    
    private EgovPasswordEncoder passwordEncoder;
    private int blockSize = 1024;	// default
   
    @Required
    public void setPasswordEncoder(EgovPasswordEncoder passwordEncoder) {
	this.passwordEncoder = passwordEncoder;
	
	logger.debug("passwordEncoder's algorithm : " + passwordEncoder.getAlgorithm());
    }
    
    public void setBlockSize(int blockSize) {
	if (blockSize % 16 != 0) {
	    blockSize += (16 - blockSize % 16);
	}
	this.blockSize = blockSize;
    }
    
    public BigDecimal encrypt(BigDecimal number, String password) {
	throw new UnsupportedOperationException("Unsupported method.. (ARIA Cryptography service doesn't support BigDecimal en/decryption)");
    }

    public byte[] encrypt(byte[] data, String password) {
	if (passwordEncoder.checkPassword(password)) {
	    ARIACipher cipher = new ARIACipher();
	    
	    cipher.setPassword(password);
	    
	    return cipher.encrypt(data);
	} else {
	    logger.error("password not matched!!!");
	    throw new IllegalArgumentException("password not matched!!!");
	}
	
    }

	public void encrypt(File srcFile, String password, File trgtFile) {
		FileInputStream fis = null;
		ByteArrayOutputStream baos = null;
		String fileString = new String();
		byte[] buffer = null;

		if (passwordEncoder.checkPassword(password)) {
			ARIACipher cipher = new ARIACipher();
			cipher.setPassword(password);
			try {
				fis = new FileInputStream(srcFile);
				baos = new ByteArrayOutputStream();
				buffer = new byte[blockSize];
				logger.debug("blockSize = {}", blockSize);
				int len = 0;
				while ((len = fis.read(buffer)) != -1) {
					if (len < blockSize) {
						byte[] tmp = new byte[len];
						System.arraycopy(buffer, 0, tmp, 0, len);
						baos.write(tmp, 0, len);
					} else {
						baos.write(buffer, 0, len);
					}
				}
				byte[] fileArray = baos.toByteArray();
				fileString = new String(Base64.encodeBase64(fileArray));
				byte[] enc = cipher.encrypt(fileString.getBytes("UTF-8"));
				String encString = Base64.encodeBase64String(enc);
				FileUtils.writeStringToFile(trgtFile, encString, "UTF-8");
			} catch (FileNotFoundException e) {
				ReflectionUtils.handleReflectionException(e);
			} catch (IOException e) {
				ReflectionUtils.handleReflectionException(e);
			} finally {
				if (fis != null) {
					try {
						fis.close();
					} catch (IOException ignore) {
					}
				}
				if (baos != null) {
					try {
						baos.close();
					} catch (IOException ignore) {
					}
				}
			}
		} else {
			logger.error("password not matched!!!");
			throw new IllegalArgumentException("password not matched!!!");
		}
	}

    /* (non-Javadoc)
     * @see egovframework.rte.fdl.cryptography.EgovCryptoService#decrypt(java.math.BigDecimal, java.lang.String)
     */
    public BigDecimal decrypt(BigDecimal encryptedNumber, String password) {
	throw new UnsupportedOperationException("Unsupported method.. (ARIA Cryptography service doesn't support BigDecimal en/decryption)");
    }

    public byte[] decrypt(byte[] encryptedData, String password) {
	if (passwordEncoder.checkPassword(password)) {
	    ARIACipher cipher = new ARIACipher();
	    
	    cipher.setPassword(password);
	    
	    return cipher.decrypt(encryptedData);
	} else {
	    logger.error("password not matched!!!");
	    throw new IllegalArgumentException("password not matched!!!");
	}
    }

	public void decrypt(File encryptedFile, String password, File trgtFile) {
		if (passwordEncoder.checkPassword(password)) {
			ARIACipher cipher = new ARIACipher();
			cipher.setPassword(password);

			try {
				String readEncString = FileUtils.readFileToString(encryptedFile, "UTF-8");
				byte[] decEnc = Base64.decodeBase64(readEncString);
				byte[] dec = cipher.decrypt(decEnc);
				String decBase64String = new String(dec, "UTF-8");
				byte[] data = Base64.decodeBase64(decBase64String);
				FileUtils.writeByteArrayToFile(trgtFile, data);
			} catch (FileNotFoundException e) {
				ReflectionUtils.handleReflectionException(e);
			} catch (IOException e) {
				ReflectionUtils.handleReflectionException(e);
			}
		} else {
			logger.error("password not matched!!!");
			throw new IllegalArgumentException("password not matched!!!");
		}
	}
}
