package egovframework.rte.cvpl.util;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

/**
 * @Class Name : EgovHsqlFunction.java
 * @Description : EgovHsqlFunction class
 * @Modification Information @ @ 수정일 수정자 수정내용 @ ------- --------
 *               --------------------------- @ 2009.03.05 천종덕 최초 생성
 * 
 * @author 실행환경 개발팀 천종덕
 * @since 2009.03.05
 * @version 1.0
 * @see Copyright (C) 2009 by MOPAS All right reserved.
 */
public class EgovHsqlFunction {

    /** 기본 DATE 포맷 */
    public static final String DATE_FORMAT = "yyyyMMdd";

    /** 기본 TimeStamp 포맷 */
    public static final String TIMESTAMP_FORMAT = "yyyyMMddHHmmss";

    /** Date Formatter */
    private static final SimpleDateFormat dateFormatter = new SimpleDateFormat(DATE_FORMAT, Locale.KOREA);

    /** TimeStamp Formatter */
    private static final SimpleDateFormat timestampFormatter = new SimpleDateFormat(TIMESTAMP_FORMAT, Locale.KOREA);

    /**
     * String을 Date형태로 바꿔준다. (기본 DATE 포맷)
     * 
     * @param str
     *            - 변경할 String
     * @return Date
     * @exception Exception
     */
    public static Date directDate(String str) throws ParseException {
        return to_date(str, DATE_FORMAT);
    }

    /**
     * String을 Date형태로 바꿔준다.
     * 
     * @param str
     *            - 변경할 String, format - 변경형식 String
     * @return Date
     * @exception Exception
     */
    public static Date to_date(String str, String format) throws ParseException {
        if (str == null || "".equals(str)) {
            return null;
        }

        if (DATE_FORMAT.equals(format)) {
            return dateFormatter.parse(str);
        } else {
            SimpleDateFormat sdf = new SimpleDateFormat(format, Locale.KOREA);
            return sdf.parse(str);
        }
    }

    /**
     * String을 Timestamp형태로 바꿔준다. (기본 TimeStamp 포맷)
     * 
     * @param str
     *            - 변경할 String
     * @return Timestamp
     * @exception Exception
     */
    public static Timestamp directTimestamp(String str) throws ParseException {
        return to_timestamp(str, TIMESTAMP_FORMAT);
    }

    /**
     * String을 Timestamp형태로 바꿔준다.
     * 
     * @param str
     *            - 변경할 String, format - 변경형식 String
     * @return Timestamp
     * @exception Exception
     */
    public static Timestamp to_timestamp(String str, String format) throws ParseException {
        if (str == null || "".equals(str)) {
            return null;
        }

        if (TIMESTAMP_FORMAT.equals(format)) {
            return new Timestamp(timestampFormatter.parse(str).getTime());
        } else {
            SimpleDateFormat sdf = new SimpleDateFormat(format, Locale.KOREA);
            return new Timestamp(sdf.parse(str).getTime());
        }
    }

    /**
     * Date를 String형태로 바꿔준다. (기본 Date 포맷)
     * 
     * @param str
     *            - 변경할 Date
     * @return String
     * @exception Exception
     */
    public static String directDateToChar(Date date) throws ParseException {
        return to_char(date, DATE_FORMAT);
    }

    /**
     * Date를 String형태로 바꿔준다.
     * 
     * @param str
     *            - 변경할 Date, format - 변경형식 String
     * @return String
     * @exception Exception
     */
    public static String to_char(Date date, String format) {
        if (date == null) {
            return null;
        }

        if (DATE_FORMAT.equals(format)) {
            return dateFormatter.format(date);
        } else {
            SimpleDateFormat sdf = new SimpleDateFormat(format);
            return sdf.format(date);
        }
    }

    /**
     * Timestamp를 String형태로 바꿔준다. (기본 TimeStamp 포맷)
     * 
     * @param str
     *            - 변경할 Timestamp
     * @return String
     * @exception Exception
     */
    public static String directTimestampToChar(Timestamp timestamp) {
        return to_char(timestamp, TIMESTAMP_FORMAT);
    }

    /**
     * Timestamp를 String형태로 바꿔준다.
     * 
     * @param str
     *            - 변경할 Timestamp, format - 변경형식 String
     * @return String
     * @exception Exception
     */
    public static String to_char(Timestamp timestamp, String format) {
        if (timestamp == null) {
            return null;
        }

        if (TIMESTAMP_FORMAT.equals(format)) {
            return timestampFormatter.format(timestamp);
        } else {
            SimpleDateFormat sdf = new SimpleDateFormat(format);
            return sdf.format(timestamp);
        }
    }
}
